"use client";

import { useEffect, useRef, useState } from "react";
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { useGSAP } from "@gsap/react";

gsap.registerPlugin(ScrollTrigger, useGSAP);

interface UseHeroAnimationProps {
    canvasRef: React.RefObject<HTMLCanvasElement | null>;
    frameCount: number;
    framePaths: string[]; // URLs or paths to images
}

export function useHeroAnimation({
    canvasRef,
    frameCount,
    framePaths,
}: UseHeroAnimationProps) {
    const frameIndexRef = useRef({ value: 0 }); // Animation target
    const imagesRef = useRef<HTMLImageElement[]>([]); // Image cache
    const [imagesLoaded, setImagesLoaded] = useState(false);

    // Preload Images
    useEffect(() => {
        if (frameCount === 0 || framePaths.length === 0) return;

        setImagesLoaded(false);
        imagesRef.current = []; // Clear cache
        let loadedCount = 0;
        let isMounted = true;

        // Parallel load
        framePaths.forEach((path, index) => {
            const img = new Image();
            img.src = path;
            img.onload = () => {
                if (!isMounted) return;
                loadedCount++;
                if (loadedCount === frameCount) {
                    setImagesLoaded(true);
                }
            };
            img.onerror = () => {
                // Handle error gracefully, count it anyway to avoid hanging
                if (!isMounted) return;
                console.error(`Failed to load frame: ${path}`);
                loadedCount++;
                if (loadedCount === frameCount) {
                    setImagesLoaded(true);
                }
            };
            imagesRef.current[index] = img;
        });

        return () => {
            isMounted = false;
        };
    }, [frameCount, framePaths]);

    // Animation Logic (runs only when images are loaded)
    useGSAP(
        () => {
            if (!imagesLoaded || !canvasRef.current || frameCount === 0) return;

            const canvas = canvasRef.current;
            const context = canvas.getContext("2d");
            if (!context) return;

            // Render Helper
            const render = () => {
                const index = Math.round(frameIndexRef.current.value);
                const safeIndex = Math.min(Math.max(index, 0), frameCount - 1);
                const img = imagesRef.current[safeIndex];

                if (img && img.complete && img.naturalHeight !== 0) {
                    // Clear
                    context.clearRect(0, 0, canvas.width, canvas.height);

                    // Aspect Ratio / Cover Logic
                    const scale = Math.max(
                        canvas.width / img.width,
                        canvas.height / img.height
                    );
                    const x = (canvas.width - img.width * scale) / 2;
                    const y = (canvas.height - img.height * scale) / 2;

                    context.drawImage(
                        img,
                        0,
                        0,
                        img.width,
                        img.height,
                        x,
                        y,
                        img.width * scale,
                        img.height * scale
                    );
                }
            };

            // Verification: Render first frame immediately
            render();

            // Check Reduced Motion
            const prefersReducedMotion = window.matchMedia("(prefers-reduced-motion: reduce)").matches;
            if (prefersReducedMotion) {
                frameIndexRef.current.value = frameCount - 1;
                render();
                return;
            }

            // --- SCROLL TRIGGER (PINNING ONLY) ---
            // This pins the section while the user scrolls, but doesn't control the frame index
            ScrollTrigger.create({
                trigger: canvas.parentElement,
                start: "top top",
                end: "+=300%", // Pin for 300vh
                pin: true,
                scrub: false, // Decoupled from animation
            });

            // --- TIME-BASED ANIMATION ---
            gsap.to(frameIndexRef.current, {
                value: frameCount - 1,
                duration: 3.5, // Cinematic duration
                ease: "power2.inOut",
                onUpdate: render,
                lazy: false, // Force immediate update if needed
            });

        },
        { scope: canvasRef, dependencies: [imagesLoaded, frameCount] }
    );

    return { imagesLoaded };
}
