"use client";

import { useState, useEffect } from "react";

interface TypewriterProps {
    texts: { content: string; isHeading?: boolean }[];
    speed?: number;
    delay?: number;
    className?: string;
    style?: React.CSSProperties;
    onComplete?: () => void;
}

export default function Typewriter({
    texts,
    speed = 50,
    delay = 800,
    className = "",
    style = {},
    onComplete,
}: TypewriterProps) {
    const [displayedTexts, setDisplayedTexts] = useState<string[]>([]);
    const [currentTextIndex, setCurrentTextIndex] = useState(0);
    const [currentCharIndex, setCurrentCharIndex] = useState(0);
    const [isTyping, setIsTyping] = useState(true);

    useEffect(() => {
        if (currentTextIndex >= texts.length) {
            setIsTyping(false);
            if (onComplete) onComplete();
            return;
        }

        const currentText = texts[currentTextIndex].content;

        if (currentCharIndex < currentText.length) {
            const timeout = setTimeout(() => {
                setDisplayedTexts((prev) => {
                    const newTexts = [...prev];
                    if (!newTexts[currentTextIndex]) {
                        newTexts[currentTextIndex] = "";
                    }
                    newTexts[currentTextIndex] = currentText.slice(0, currentCharIndex + 1);
                    return newTexts;
                });
                setCurrentCharIndex((prev) => prev + 1);
            }, speed);

            return () => clearTimeout(timeout);
        } else {
            // Move to next text after delay
            const timeout = setTimeout(() => {
                setCurrentTextIndex((prev) => prev + 1);
                setCurrentCharIndex(0);
            }, delay);

            return () => clearTimeout(timeout);
        }
    }, [currentCharIndex, currentTextIndex, texts, speed, delay, onComplete]);

    return (
        <div className={className} style={style}>
            {displayedTexts.map((text, index) => {
                const isHeading = texts[index]?.isHeading;
                const isCurrentlyTyping = isTyping && index === currentTextIndex;

                return (
                    <div
                        key={index}
                        style={{
                            marginBottom: isHeading ? "-20px" : "var(--space-xl)",
                            position: "relative",
                        }}
                    >
                        {isHeading ? (
                            <span style={{ fontWeight: 700 }}>
                                {text}
                            </span>
                        ) : (
                            <span className="body-lg" style={{ opacity: 0.85, fontSize: "var(--text-body-lg)", lineHeight: "var(--leading-relaxed)" }}>
                                {text}
                            </span>
                        )}
                        {isCurrentlyTyping && (
                            <span className="typewriter-cursor" />
                        )}
                    </div>
                );
            })}

            <style jsx>{`
        .typewriter-cursor {
          display: inline-block;
          width: 3px;
          height: 0.9em;
          background: linear-gradient(180deg, #1e3a8a 0%, #0f172a 100%);
          margin-left: 4px;
          animation: blink 1s step-end infinite;
          vertical-align: baseline;
          position: relative;
          top: 0.1em;
        }

        @keyframes blink {
          0%, 50% {
            opacity: 1;
          }
          51%, 100% {
            opacity: 0;
          }
        }
      `}</style>
        </div>
    );
}
