"use client";

import { useRef } from "react";
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { useGSAP } from "@gsap/react";

gsap.registerPlugin(ScrollTrigger, useGSAP);

interface ServiceCard {
    title: string;
    description: string;
    icon: string;
    size: "small" | "wide" | "tall" | "large";
}

const services: ServiceCard[] = [
    {
        title: "Fullstack Systems",
        description: "End-to-end digital infrastructure. From robust backends to intuitive frontends, we architect complete systems that scale with your business.",
        icon: "",
        size: "large",
    },
    {
        title: "Mobile Development",
        description: "Native and cross-platform mobile applications. We build responsive, high-performance apps for iOS and Android that deliver exceptional user experiences.",
        icon: "",
        size: "tall",
    },
    {
        title: "Web Development",
        description: "Modern, dynamic web applications. We utilize cutting-edge frameworks to create fast, SEO-optimized, and visually stunning websites.",
        icon: "",
        size: "tall",
    },
    {
        title: "Maintenance & Support",
        description: "Long-term reliability. We provide ongoing support, updates, and optimization to ensure your digital assets remain secure and performant.",
        icon: "",
        size: "wide",
    },
];

export default function Services() {
    const sectionRef = useRef<HTMLElement>(null);
    const cardsRef = useRef<HTMLDivElement>(null);

    useGSAP(
        () => {
            const cards = cardsRef.current?.querySelectorAll(".service-card");
            if (!cards) return;

            gsap.fromTo(
                cards,
                {
                    opacity: 0,
                    y: 60,
                },
                {
                    opacity: 1,
                    y: 0,
                    duration: 1.2,
                    stagger: 0.15,
                    ease: "power2.out",
                    scrollTrigger: {
                        trigger: sectionRef.current,
                        start: "top 70%",
                        toggleActions: "play none none reverse",
                    },
                }
            );
        },
        { scope: sectionRef }
    );

    return (
        <section
            id="services"
            ref={sectionRef}
            className="section"
            style={{
                paddingTop: "var(--space-5xl)",
                paddingBottom: "var(--space-5xl)",
            }}
        >
            <div className="container">
                <div
                    className="service-parent-card animated-gradient-card"
                    style={{
                        borderRadius: "var(--radius-lg)",
                        padding: "var(--space-4xl)",
                        position: "relative",
                        overflow: "hidden",
                    }}
                >
                    {/* Corner Borders */}
                    <div className="corner-borders"></div>

                    {/* Section Header */}
                    <div style={{ textAlign: "center", marginBottom: "var(--space-4xl)" }}>
                        <h2 className="heading-xl" style={{ marginBottom: "var(--space-lg)" }}>
                            Our Services
                        </h2>
                        <p className="body-lg" style={{ maxWidth: "50ch", margin: "0 auto", opacity: 0.85 }}>
                            Comprehensive technical solutions tailored to your unique requirements.
                        </p>
                    </div>

                    {/* Services Grid */}
                    <div
                        ref={cardsRef}
                        className="service-grid"
                        style={{
                            display: "grid",
                            gridTemplateColumns: "repeat(12, 1fr)",
                            gap: "var(--space-xl)",
                            maxWidth: "1400px",
                            margin: "0 auto",
                        }}
                    >
                        {services.map((service, index) => {
                            let gridColumn = "";
                            let gridRow = "";

                            // Layout logic
                            if (service.size === "large") {
                                gridColumn = "1 / 13"; // Full width top
                                gridRow = "1 / 2";
                            } else if (service.size === "tall") {
                                // Fullstack is large (top), Mobile (left), Web (right), Maintenance (bottom)
                                // Adjusting based on array order:
                                // 0: Large (Fullstack) -> Row 1
                                // 1: Tall (Mobile) -> Row 2 Left
                                // 2: Tall (Web) -> Row 2 Right
                                // 3: Wide (Maintenance) -> Row 3

                                if (index === 1) gridColumn = "1 / 7";
                                if (index === 2) gridColumn = "7 / 13";
                                gridRow = "2 / 3";
                            } else if (service.size === "wide") {
                                gridColumn = "1 / 13";
                                gridRow = "3 / 4";
                            }

                            return (
                                <div
                                    key={index}
                                    className="service-card"
                                    style={{
                                        background: "var(--bg-surface)",
                                        borderRadius: "var(--radius-lg)",
                                        padding: "var(--space-2xl)",
                                        position: "relative",
                                        overflow: "hidden",
                                        gridColumn,
                                        gridRow,
                                        display: "flex",
                                        flexDirection: "column",
                                        justifyContent: "space-between",
                                        minHeight: service.size === "tall" ? "400px" : "300px",
                                    }}
                                >
                                    {/* Corner Borders for inner cards too? Original didn't have them on all inner cards, but 'industry-card' did. Let's add them for consistency if desired, or keep clean. The prompt says "match the structure to the existing sections". IndustrySolutions has corner borders on inner cards. */}
                                    <div className="corner-borders"></div>

                                    <div>
                                        <div style={{
                                            fontSize: "3rem",
                                            marginBottom: "var(--space-lg)",
                                            display: "inline-block"
                                        }}>
                                            {service.icon}
                                        </div>
                                        <h3
                                            className="heading-md"
                                            style={{
                                                marginBottom: "var(--space-md)",
                                                fontSize: "var(--text-heading-md)",
                                            }}
                                        >
                                            {service.title}
                                        </h3>
                                        <p className="body-lg" style={{ opacity: 0.85 }}>
                                            {service.description}
                                        </p>
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                </div>
            </div>

            {/* Corner Border Styles - Including white override for dark mode consistency */}
            <style jsx>{`
                .corner-borders::before,
                .corner-borders::after {
                    content: "";
                    position: absolute;
                    width: 50px;
                    height: 50px;
                    border: 2px solid #171717;
                }

                [data-theme="dark"] .corner-borders::before,
                [data-theme="dark"] .corner-borders::after {
                    border-color: #FFFFFF !important;
                }

                .corner-borders::before {
                    top: 0;
                    left: 0;
                    border-right: none;
                    border-bottom: none;
                    border-top-left-radius: var(--radius-lg);
                }

                .corner-borders::after {
                    bottom: 0;
                    right: 0;
                    border-left: none;
                    border-top: none;
                    border-bottom-right-radius: var(--radius-lg);
                }

                .service-card {
                    transition: transform 0.3s ease, box-shadow 0.3s ease;
                }

                .service-card:hover {
                    transform: translateY(-4px);
                    box-shadow: 0 12px 32px rgba(0, 0, 0, 0.08);
                }

                .service-parent-card {
                    transition: transform 0.3s ease, box-shadow 0.3s ease;
                }

                .service-parent-card:hover {
                    transform: translateY(-4px);
                    box-shadow: 0 12px 32px rgba(0, 0, 0, 0.08);
                }

                @media (max-width: 1024px) {
                    .service-grid {
                        display: flex !important;
                        flex-direction: column !important;
                    }
                    
                    .corner-borders::before,
                    .corner-borders::after {
                        width: 40px;
                        height: 40px;
                    }
                }

                @media (max-width: 767px) {
                    .corner-borders::before,
                    .corner-borders::after {
                        width: 30px;
                        height: 30px;
                    }
                }
            `}</style>
        </section>
    );
}
