"use client";

import { useState, useRef, useEffect } from "react";
import gsap from "gsap";
import { useGSAP } from "@gsap/react";

interface Option {
    label: string;
    value: string;
}

interface ServiceDropdownProps {
    options: Option[];
    value: string;
    onChange: (value: string) => void;
    placeholder?: string;
    id?: string;
}

export default function ServiceDropdown({ options, value, onChange, placeholder = "Select an option", id }: ServiceDropdownProps) {
    const [isOpen, setIsOpen] = useState(false);
    const containerRef = useRef<HTMLDivElement>(null);
    const listRef = useRef<HTMLDivElement>(null);
    const chevronRef = useRef<SVGSVGElement>(null);

    // Close on click outside
    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (containerRef.current && !containerRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };

        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    const { contextSafe } = useGSAP({ scope: containerRef });

    const toggleOpen = contextSafe(() => {
        setIsOpen(!isOpen);
    });

    const handleSelect = (optionValue: string) => {
        onChange(optionValue);
        setIsOpen(false);
    };

    // Animation for list opening/closing
    useGSAP(() => {
        if (isOpen) {
            gsap.to(listRef.current, {
                height: "auto",
                opacity: 1,
                duration: 0.4,
                ease: "power2.out",
                display: "block"
            });
            gsap.to(chevronRef.current, {
                rotation: 180,
                duration: 0.3,
                ease: "power2.out"
            });
        } else {
            gsap.to(listRef.current, {
                height: 0,
                opacity: 0,
                duration: 0.3,
                ease: "power2.in",
                display: "none"
            });
            gsap.to(chevronRef.current, {
                rotation: 0,
                duration: 0.3,
                ease: "power2.out"
            });
        }
    }, { dependencies: [isOpen], scope: containerRef });

    const selectedLabel = options.find((opt) => opt.value === value)?.label || placeholder;
    const isPlaceholder = !value;

    return (
        <div
            ref={containerRef}
            id={id}
            style={{
                position: "relative",
                width: "100%",
                fontFamily: "inherit"
            }}
        >
            {/* Trigger Button */}
            <button
                type="button"
                onClick={toggleOpen}
                style={{
                    width: "100%",
                    padding: "var(--space-md)",
                    borderRadius: "var(--radius-md)",
                    border: "1px solid var(--border-subtle)",
                    background: "var(--bg-surface)",
                    color: isPlaceholder ? "var(--text-tertiary)" : "var(--text-primary)", // Visually lighter if placeholder
                    fontSize: "1rem",
                    textAlign: "left",
                    cursor: "pointer",
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                    transition: "border-color 0.2s ease, box-shadow 0.2s ease",
                    outline: "none"
                }}
                onFocus={(e) => e.currentTarget.style.borderColor = "var(--text-primary)"}
                onBlur={(e) => !isOpen && (e.currentTarget.style.borderColor = "var(--border-subtle)")}
            >
                <span style={{ overflow: "hidden", textOverflow: "ellipsis", whiteSpace: "nowrap" }}>
                    {selectedLabel}
                </span>
                <svg
                    ref={chevronRef}
                    width="20"
                    height="20"
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    style={{ marginLeft: "12px", flexShrink: 0, color: "var(--text-secondary)" }}
                >
                    <polyline points="6 9 12 15 18 9"></polyline>
                </svg>
            </button>

            {/* Dropdown List */}
            <div
                ref={listRef}
                style={{
                    position: "absolute",
                    top: "calc(100% + 8px)",
                    left: 0,
                    width: "100%",
                    background: "var(--bg-surface)",
                    border: "1px solid var(--border-subtle)",
                    borderRadius: "var(--radius-md)",
                    boxShadow: "0 4px 20px rgba(0, 0, 0, 0.08)",
                    zIndex: 20,
                    overflow: "hidden",
                    height: 0,
                    opacity: 0,
                    display: "none"
                }}
            >
                <ul style={{ listStyle: "none", padding: "8px 0", margin: 0, maxHeight: "300px", overflowY: "auto" }}>
                    {options.map((option) => (
                        <li key={option.value}>
                            <button
                                type="button"
                                onClick={() => handleSelect(option.value)}
                                style={{
                                    width: "100%",
                                    padding: "12px var(--space-md)",
                                    textAlign: "left",
                                    background: "transparent",
                                    border: "none",
                                    color: value === option.value ? "var(--text-primary)" : "var(--text-secondary)",
                                    fontWeight: value === option.value ? 500 : 400,
                                    cursor: "pointer",
                                    fontSize: "1rem",
                                    transition: "background 0.2s ease, color 0.2s ease"
                                }}
                                onMouseEnter={(e) => {
                                    e.currentTarget.style.background = "var(--border-light)";
                                    e.currentTarget.style.color = "var(--text-primary)";
                                }}
                                onMouseLeave={(e) => {
                                    e.currentTarget.style.background = "transparent";
                                    // Reset color only if not selected
                                    if (value !== option.value) {
                                        e.currentTarget.style.color = "var(--text-secondary)";
                                    }
                                }}
                            >
                                {option.label}
                            </button>
                        </li>
                    ))}
                </ul>
            </div>
        </div>
    );
}
