"use client";

import { useRef } from "react";
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { useGSAP } from "@gsap/react";

gsap.registerPlugin(ScrollTrigger, useGSAP);

interface IndustryCard {
    title: string;
    description: string;
    category: string;
    size: "small" | "wide" | "tall" | "large";
}

const industries: IndustryCard[] = [
    {
        category: "Industry & Enterprise SaaS",
        title: "Scalable software solutions for enterprise needs.",
        description: "Built for performance, reliability, and seamless integration.",
        size: "tall",
    },
    {
        category: "Fast Delivery & Low Cost",
        title: "Accelerated deployment cycles and cost-effective solutions for your business.",
        description: "Optimized workflows and efficient resource management.",
        size: "small",
    },
    {
        category: "Healthcare & Medical",
        title: "HIPAA-compliant solutions for modern healthcare delivery.",
        description: "Secure patient data management, telemedicine platforms, and clinical workflow optimization designed for healthcare providers.",
        size: "wide",
    },
    {
        category: "Finance & FinTech",
        title: "Secure, compliant financial technology solutions.",
        description: "Payment processing, fraud detection, and regulatory compliance tools built for the financial sector.",
        size: "large",
    },
];

export default function IndustrySolutions() {
    const sectionRef = useRef<HTMLElement>(null);
    const cardsRef = useRef<HTMLDivElement>(null);

    useGSAP(
        () => {
            const cards = cardsRef.current?.querySelectorAll(".industry-card");
            if (!cards) return;

            gsap.fromTo(
                cards,
                {
                    opacity: 0,
                    y: 60,
                },
                {
                    opacity: 1,
                    y: 0,
                    duration: 1.2,
                    stagger: 0.15,
                    ease: "power2.out",
                    scrollTrigger: {
                        trigger: sectionRef.current,
                        start: "top 70%",
                        toggleActions: "play none none reverse",
                    },
                }
            );
        },
        { scope: sectionRef }
    );

    return (
        <section
            ref={sectionRef}
            className="section"
            style={{
                paddingTop: "var(--space-5xl)",
                paddingBottom: "var(--space-5xl)",
            }}
        >
            <div className="container">
                <div
                    className="industry-parent-card animated-gradient-card"
                    style={{
                        borderRadius: "var(--radius-lg)",
                        padding: "var(--space-4xl)",
                        position: "relative",
                        overflow: "hidden",
                    }}
                >
                    {/* Corner Borders */}
                    <div className="corner-borders"></div>

                    {/* Section Header */}
                    <div style={{ textAlign: "center", marginBottom: "var(--space-4xl)" }}>
                        <h2 className="heading-xl" style={{ marginBottom: "var(--space-lg)" }}>
                            Tailored Solutions for Every Industry
                        </h2>
                        <p className="body-lg" style={{ maxWidth: "50ch", margin: "0 auto", opacity: 0.85 }}>
                            Specialized software solutions designed to meet the unique challenges of your industry.
                        </p>
                    </div>

                    {/* Industry Cards Grid - Asymmetric Layout */}
                    <div
                        ref={cardsRef}
                        className="industry-grid"
                        style={{
                            display: "grid",
                            gridTemplateColumns: "repeat(12, 1fr)",
                            gap: "var(--space-xl)",
                            maxWidth: "1400px",
                            margin: "0 auto",
                        }}
                    >
                        {industries.map((industry, index) => {
                            let gridColumn = "";
                            let gridRow = "";

                            // Create asymmetric layout
                            if (industry.size === "tall") {
                                gridColumn = "1 / 7"; // Left half
                                gridRow = "1 / 3"; // Spans 2 rows
                            } else if (industry.size === "small") {
                                gridColumn = "7 / 13"; // Right half
                                gridRow = "1 / 2"; // Single row
                            } else if (industry.size === "wide") {
                                gridColumn = "7 / 13"; // Right half
                                gridRow = "2 / 3"; // Second row
                            } else if (industry.size === "large") {
                                gridColumn = "1 / 13"; // Full width
                                gridRow = "3 / 4"; // Third row
                            }

                            return (
                                <div
                                    key={index}
                                    className="industry-card"
                                    style={{
                                        background: "var(--bg-surface)",
                                        borderRadius: "var(--radius-lg)",
                                        padding: "var(--space-2xl)",
                                        position: "relative",
                                        overflow: "hidden",
                                        gridColumn,
                                        gridRow,
                                        display: "flex",
                                        flexDirection: "column",
                                        justifyContent: "space-between",
                                    }}
                                >
                                    {/* Corner Borders */}
                                    <div className="corner-borders"></div>

                                    {/* Video for the tall card */}
                                    {industry.title === "Scalable software solutions for enterprise needs." && (
                                        <div style={{
                                            flex: 1,
                                            marginBottom: "var(--space-lg)",
                                            borderRadius: "var(--radius-md)",
                                            overflow: "hidden",
                                            position: "relative",
                                            minHeight: "200px"
                                        }}>
                                            <video
                                                autoPlay
                                                muted
                                                loop
                                                playsInline
                                                src="/solutions-video.mp4"
                                                style={{ width: "100%", height: "100%", objectFit: "cover" }}
                                            />
                                        </div>
                                    )}

                                    {/* Content */}
                                    <div>
                                        <p
                                            className="caption"
                                            style={{
                                                marginBottom: "var(--space-md)",
                                                color: "var(--text-tertiary)",
                                            }}
                                        >
                                            {industry.category}
                                        </p>
                                        <h3
                                            className="heading-md"
                                            style={{
                                                marginBottom: "var(--space-md)",
                                                fontSize: industry.size === "large" ? "var(--text-heading-lg)" : "var(--text-heading-md)",
                                            }}
                                        >
                                            {industry.title}
                                        </h3>
                                        <p className="body-lg" style={{ opacity: 0.85 }}>
                                            {industry.description}
                                        </p>
                                    </div>
                                </div>
                            );
                        })}
                    </div>

                    {/* Bottom Feature Box */}
                    <div
                        className="industry-card"
                        style={{
                            background: "var(--bg-surface)",
                            borderRadius: "var(--radius-lg)",
                            padding: "var(--space-3xl)",
                            marginTop: "var(--space-3xl)",
                            textAlign: "center",
                            position: "relative",
                            overflow: "hidden",
                        }}
                    >
                        {/* Corner Borders */}
                        <div className="corner-borders"></div>

                        <h3 className="heading-lg" style={{ marginBottom: "var(--space-lg)" }}>
                            Enterprise-grade maintenance and AI-powered solutions for reliability and innovation.
                        </h3>

                        {/* Feature Labels */}
                        <div
                            style={{
                                display: "flex",
                                justifyContent: "center",
                                gap: "var(--space-3xl)",
                                marginTop: "var(--space-2xl)",
                                flexWrap: "wrap",
                            }}
                        >
                            {["24/7 Reliability", "AI Innovation", "Security First", "Cloud Scale"].map((feature, idx) => (
                                <div key={idx} style={{ textAlign: "center" }}>
                                    <div
                                        style={{
                                            width: "80px",
                                            height: "80px",
                                            borderRadius: "50%",
                                            background: "var(--bg-page)",
                                            border: "2px solid var(--border-subtle)",
                                            display: "flex",
                                            alignItems: "center",
                                            justifyContent: "center",
                                            margin: "0 auto var(--space-sm)",
                                        }}
                                    >
                                        <span style={{ fontSize: "var(--text-body)", fontWeight: 500 }}>
                                            {idx + 1}
                                        </span>
                                    </div>
                                    <p className="body" style={{ color: "var(--text-secondary)", fontWeight: 500 }}>
                                        {feature}
                                    </p>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            </div>

            {/* Corner Border Styles */}
            <style jsx>{`
        .corner-borders::before,
        .corner-borders::after {
          content: "";
          position: absolute;
          width: 50px;
          height: 50px;
          border: 2px solid #171717;
        }

        /* Top-left corner */
        .corner-borders::before {
          top: 0;
          left: 0;
          border-right: none;
          border-bottom: none;
          border-top-left-radius: var(--radius-lg);
        }

        /* Bottom-right corner */
        .corner-borders::after {
          bottom: 0;
          right: 0;
          border-left: none;
          border-top: none;
          border-bottom-right-radius: var(--radius-lg);
        }

        .industry-card {
          transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .industry-card:hover {
          transform: translateY(-4px);
          box-shadow: 0 12px 32px rgba(0, 0, 0, 0.08);
        }

        .industry-parent-card {
          transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .industry-parent-card:hover {
          transform: translateY(-4px);
          box-shadow: 0 12px 32px rgba(0, 0, 0, 0.08);
        }

        @media (max-width: 1024px) {
          .industry-grid {
            display: flex !important;
            flex-direction: column !important;
          }
          
          .corner-borders::before,
          .corner-borders::after {
            width: 40px;
            height: 40px;
          }
        }

        @media (max-width: 767px) {
          .corner-borders::before,
          .corner-borders::after {
            width: 30px;
            height: 30px;
          }
        }
      `}</style>
        </section>
    );
}
