"use client";

import { useEffect, useRef, useState } from "react";
import { useHeroAnimation } from "@/hooks/useHeroAnimation";

interface HeroSequenceMetadata {
    count: number;
    frames: string[];
}

export default function HeroChip() {
    const canvasRef = useRef<HTMLCanvasElement>(null);
    const containerRef = useRef<HTMLElement>(null);
    const [metadata, setMetadata] = useState<HeroSequenceMetadata | null>(null);

    // Metadata loading state
    const [metadataLoaded, setMetadataLoaded] = useState(false);

    useEffect(() => {
        async function fetchMetadata() {
            try {
                const res = await fetch("/api/hero-sequence-metadata");
                if (!res.ok) throw new Error("Failed to fetch metadata");
                const data = await res.json();
                setMetadata(data);
                setMetadataLoaded(true);
            } catch (error) {
                console.error(error);
                setMetadataLoaded(true); // Fail open? Or handle error
            }
        }
        fetchMetadata();
    }, []);

    // Frame paths from metadata
    const framePaths = metadata?.frames || [];

    // Hook handles image preloading + animation
    // Now returns loading state
    const { imagesLoaded } = useHeroAnimation({
        canvasRef,
        frameCount: metadata?.count || 0,
        framePaths,
    });

    // Global loading state: Metadata + Images
    const isLoading = !metadataLoaded || !imagesLoaded;

    return (
        <section
            ref={containerRef}
            className="relative w-full"
            style={{ height: "400vh" }} // Scroll space for pinning
        >
            <div className="sticky top-0 left-0 w-full h-screen flex items-center justify-center overflow-hidden bg-black">

                {/* Loading Overlay */}
                {isLoading && (
                    <div className="absolute inset-0 flex items-center justify-center bg-black z-50">
                        <div className="flex flex-col items-center">
                            <div className="w-12 h-12 border-2 border-white/20 border-t-white rounded-full animate-spin mb-4" />
                            <p className="text-white/50 text-sm tracking-widest uppercase animate-pulse">
                                Initializing Core...
                            </p>
                        </div>
                    </div>
                )}

                <canvas
                    ref={canvasRef}
                    className="block w-full h-full object-cover"
                    width={1920}
                    height={1080}
                />

                {/* Minimal Overlay - Fade in when loaded */}
                {/* Using simple opacity transition for cleanliness, no extra deps */}
                <div
                    className={`absolute inset-0 flex flex-col items-center justify-center pointer-events-none z-20 transition-opacity duration-1000 ${isLoading ? 'opacity-0' : 'opacity-100'}`}
                >
                    <h1 className="text-4xl md:text-6xl font-bold text-white tracking-tighter mix-blend-difference">
                        SANVYA
                    </h1>
                    <p className="text-white/70 text-sm tracking-widest mt-4 uppercase">
                        Technology Redefined
                    </p>
                </div>
            </div>
        </section>
    );
}
