"use client";

import { useEffect, useRef } from "react";
import { useTheme } from "@/contexts/ThemeContext";

export default function AnimatedBackground() {
    const canvasRef = useRef<HTMLCanvasElement>(null);
    const { theme } = useTheme();

    useEffect(() => {
        const canvas = canvasRef.current;
        if (!canvas) return;

        const ctx = canvas.getContext("2d");
        if (!ctx) return;

        // Set canvas size
        const resizeCanvas = () => {
            canvas.width = window.innerWidth;
            canvas.height = window.innerHeight;
        };
        resizeCanvas();
        window.addEventListener("resize", resizeCanvas);

        // Particle configuration
        const particles: Array<{
            x: number;
            y: number;
            z: number;
            baseX: number;
            baseY: number;
            size: number;
            opacity: number;
            opacityDirection: number;
            speed: number;
        }> = [];

        const particleCount = 100;

        // Dark mode: pure white with 35% opacity, Light mode: blue colors
        const colors = theme === "dark"
            ? [
                "rgba(255, 255, 255, ", // Pure white
                "rgba(255, 255, 255, ", // Pure white
                "rgba(255, 255, 255, ", // Pure white
            ]
            : [
                "rgba(59, 130, 246, ", // Blue-500
                "rgba(96, 165, 250, ", // Blue-400
                "rgba(37, 99, 235, ",  // Blue-600
            ];

        // Create particles
        for (let i = 0; i < particleCount; i++) {
            const baseOpacity = theme === "dark" ? 0.35 : (Math.random() * 0.5 + 0.2);
            particles.push({
                x: Math.random() * canvas.width,
                y: Math.random() * canvas.height,
                z: Math.random() * 3, // Depth for 3D effect
                baseX: Math.random() * canvas.width,
                baseY: Math.random() * canvas.height,
                size: Math.random() * 3 + 1,
                opacity: baseOpacity,
                opacityDirection: Math.random() > 0.5 ? 1 : -1,
                speed: Math.random() * 0.3 + 0.1,
            });
        }

        let mouseX = 0;
        let mouseY = 0;

        // Mouse move handler for parallax
        const handleMouseMove = (e: MouseEvent) => {
            mouseX = e.clientX;
            mouseY = e.clientY;
        };
        window.addEventListener("mousemove", handleMouseMove);

        // Animation loop
        const animate = () => {
            ctx.clearRect(0, 0, canvas.width, canvas.height);

            particles.forEach((particle) => {
                // Parallax effect based on mouse position and depth
                const parallaxX = (mouseX - canvas.width / 2) * particle.z * 0.01;
                const parallaxY = (mouseY - canvas.height / 2) * particle.z * 0.01;

                // Floating animation
                particle.x = particle.baseX + Math.sin(Date.now() * 0.001 * particle.speed) * 30 + parallaxX;
                particle.y = particle.baseY + Math.cos(Date.now() * 0.001 * particle.speed) * 30 + parallaxY;

                // Blinking effect
                particle.opacity += particle.opacityDirection * 0.005;
                if (particle.opacity >= 0.7 || particle.opacity <= 0.1) {
                    particle.opacityDirection *= -1;
                }

                // Draw particle with glow effect
                const gradient = ctx.createRadialGradient(
                    particle.x,
                    particle.y,
                    0,
                    particle.x,
                    particle.y,
                    particle.size * 3
                );

                const colorIndex = Math.floor(particle.z);
                const color = colors[colorIndex] || colors[0];

                gradient.addColorStop(0, color + particle.opacity + ")");
                gradient.addColorStop(0.5, color + (particle.opacity * 0.5) + ")");
                gradient.addColorStop(1, color + "0)");

                ctx.fillStyle = gradient;
                ctx.beginPath();
                ctx.arc(particle.x, particle.y, particle.size * 2, 0, Math.PI * 2);
                ctx.fill();

                // Draw connecting lines between nearby particles
                particles.forEach((otherParticle) => {
                    const dx = particle.x - otherParticle.x;
                    const dy = particle.y - otherParticle.y;
                    const distance = Math.sqrt(dx * dx + dy * dy);

                    if (distance < 150) {
                        const lineColor = theme === "dark"
                            ? `rgba(255, 255, 255, ${(1 - distance / 150) * 0.35})`
                            : `rgba(59, 130, 246, ${(1 - distance / 150) * 0.1})`;
                        ctx.strokeStyle = lineColor;
                        ctx.lineWidth = 0.5;
                        ctx.beginPath();
                        ctx.moveTo(particle.x, particle.y);
                        ctx.lineTo(otherParticle.x, otherParticle.y);
                        ctx.stroke();
                    }
                });
            });

            requestAnimationFrame(animate);
        };

        animate();

        return () => {
            window.removeEventListener("resize", resizeCanvas);
            window.removeEventListener("mousemove", handleMouseMove);
        };
    }, []);

    return (
        <canvas
            ref={canvasRef}
            style={{
                position: "absolute",
                top: 0,
                left: 0,
                width: "100%",
                height: "100%",
                pointerEvents: "none",
                zIndex: 0,
            }}
        />
    );
}
